/*
    plm.c - common functions for plmtools
    Copyright (2012) University of Massachusetts Amherst
    Contact Sean Barker (sbarker@cs.umass.edu)
    Copyright (C) 2008  Matthew Randolph
    See the file COPYING for license information.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "plm.h"

int ishex(char c) {
	signed char uc = toupper(c);
  return (((uc >= '0') && (uc <= '9')) || ((uc >= 'A') && (uc <= 'F')));
}

void outhex(int fdes, char c) {
	unsigned char uc;
	signed char tc;
	uc = *(unsigned char *)(&c);
	tc = ((uc / 16) % 16);
	tc = ((tc >= 0) && (tc <= 9)) ? (tc + '0') : (tc - 10 + 'A');
	write(fdes,&tc,1);
	tc = (uc % 16);
	tc = ((tc >= 0) && (tc <= 9)) ? (tc + '0') : (tc - 10 + 'A');
	write(fdes,&tc,1);
}

char hextoc(char c1, char c2) {
	unsigned char uc;
	c1 = toupper(c1);
	c2 = toupper(c2);
	if ((c1 >= '0') && (c1 <= '9')) {
		uc = c1 - '0';
  }	else {
		uc = c1 - 'A' + 10;
  }
	uc *= 16;
	if ((c2 >= '0') && (c2 <= '9')) {
		uc += c2 - '0';
  }	else {
		uc += c2 - 'A' + 10;
  }
	return *(char *)(&uc);
}

char* cmdtype(unsigned char cmd) {
  switch (cmd) {
    case 0x10:
      return "ping";
    case 0x11:
      return "on";
    case 0x12:
      return "fast on";
    case 0x13:
      return "off";
    case 0x14:
      return "fast off";
    case 0x15:
      return "bright";
    case 0x16:
      return "dim";
    case 0x17:
      return "start manual change";
    case 0x18:
      return "stop manual change";
    case 0x19:
      return "status request";
    case 0x82:
      return "meter query";
    default:
      // this may not always be true?
      return "data";
  }
}

int readpacket(int ttyfd, int nfds, fd_set* readfds, struct timespec* sleeptime, int print) {
  char buf[100];
  char* type = NULL;
  int packetlen = 0, bytesread = 0, c;
  pselect(nfds, readfds, NULL, NULL, sleeptime, NULL);
  while (bytesread < 2 && 0 < read(ttyfd, &c, 1)) {
    buf[bytesread++] = c;
  }
  if (bytesread) {
    // look for standard message starter
    if (bytesread == 2 && buf[0] == 0x02) {
      switch (buf[1]) {
        case 0x50:
          type = "standard msg";
          packetlen = 11;
          break;
        case 0x51:
          // e.g. iMeter data
          type = "extended msg";
          packetlen = 25;
          break;
        case 0x62:
          type = "ack";
          // 8 byte cmd echoed back plus 0x06 on the end
          packetlen = 9;
          break;
        case 0x54:
          type = "pairing announce/confirm";
          packetlen = 3;
          break;
        case 0x53:
          type = "pairing";
          packetlen = 10;
          break;
      }
    }
    if (!packetlen) {
      // don't know what we're reading - just read everything available
      packetlen = 100;
    }
    while (bytesread < packetlen && 0 < read(ttyfd, &c, 1)) {
      buf[bytesread++] = c;
    }

    if (print) {
      for (c = 0; c < bytesread; c++) {
        outhex(STDOUT, buf[c]);
      }
      if (print == PRINT_DEBUG) {
        if (bytesread == packetlen) {
          fprintf(stderr, "\t\t(%s", type);
          if (buf[1] == 0x50 || buf[1] == 0x62) {
            char cmd = buf[packetlen - (buf[1] == 0x50 ? 2 : 3)];
            char arg = buf[packetlen - (buf[1] == 0x50 ? 1 : 2)];
            fprintf(stderr, ", %s, arg ", cmdtype(cmd));
            outhex(STDERR, arg);
          }
          fprintf(stderr, ")");
        } else if (packetlen == 100) {
          fprintf(stderr, " (failed to parse packet structure)");
        } else {
          fprintf(stderr, " (incomplete %s, may be corrupted)", type);
        }
      }
      c = '\n';
      write(STDOUT, &c, 1);
    }

    // some weird ack stuff below
    /*
    if (packetlen == 9) // there is no ACK if we're talking to the PLM itself
      fprintf(stderr, "\nverbose printing\n");
    while ((c != 0x06) && (c != 0x15)) {
      read(ttyfd,&c,1);
      fprintf(stderr, "\n read %x\n", c);
    }
    */
  }
  return bytesread;
}

void writepacket(int ttyfd, char* cmd) {
  char* cp;
  char c;
  for (cp = cmd; ishex(*cp) && ishex(*(cp + 1)); cp += 2) {
    c = hextoc(*cp, *(cp + 1));
    write(ttyfd, &c, 1);
    // read(ttyfd,&c,1);
    //printf("%.2X",(uc)c);
  }
}

int ttysetup(char* ttydevice, fd_set* readfds) {
  int ttyfd = open(ttydevice, O_RDWR | O_NOCTTY);
  struct termios newtio;
	if (ttyfd == -1) {
//		fprintf(stderr,"%s ERROR: ",argv[0]);
		switch (errno) {
      case EACCES: case ENODEV: case ENOENT: case ENXIO:
        fprintf(stderr,"cannot open %s\n", ttydevice);
        exit(1);
        break;
      default:
        fprintf(stderr, "unknown error\n");
        exit(2);
        break;
    }
	}
	FD_ZERO(readfds);
	tcgetattr(ttyfd, &newtio);
	newtio.c_cflag = BAUDRATE | HWFLOWCTRL | LINEBITS | LINESTOPBITS | LINEPARITY | CLOCAL | CREAD;
	newtio.c_iflag = IGNBRK | IGNPAR;
	newtio.c_oflag = ONLRET;
  newtio.c_lflag = 0;
	newtio.c_cc[VMIN] = 0;
	newtio.c_cc[VTIME] = 1;
  tcflush(ttyfd, TCIFLUSH);
	tcsetattr(ttyfd, TCSANOW, &newtio);
  return ttyfd;
}

